import './styles.css';

// Animate counter values
function animateValue(element: HTMLElement, start: number, end: number, duration: number) {
    const startTime = performance.now();
    const isFloat = end.toString().includes('.');

    function update(currentTime: number) {
        const elapsed = currentTime - startTime;
        const progress = Math.min(elapsed / duration, 1);

        const easeOutQuad = 1 - (1 - progress) * (1 - progress);
        const current = start + (end - start) * easeOutQuad;

        if (isFloat) {
            element.textContent = current.toFixed(1);
        } else {
            element.textContent = Math.floor(current).toLocaleString('tr-TR');
        }

        if (progress < 1) {
            requestAnimationFrame(update);
        }
    }

    requestAnimationFrame(update);
}

// Initialize summary card animations
function initializeSummaryCards() {
    const summaryValues = document.querySelectorAll('.summary-value');

    for (const element of summaryValues) {
        const target = Number.parseFloat(element.getAttribute('data-target') || '0');
        animateValue(element as HTMLElement, 0, target, 2000);
    }
}

// Generate sample table data
interface TableRow {
    userId: string;
    activityType: string;
    time: string;
    status: 'active' | 'inactive' | 'pending';
    duration: string;
}

function generateTableData(): TableRow[] {
    const activities = [
        'Veri Görüntüleme',
        'Rapor Oluşturma',
        'Dashboard Erişimi',
        'Ayar Değişikliği',
        'Analiz İşlemi',
        'Veri Dışa Aktarma',
        'Grafik Görüntüleme',
        'Filtreleme İşlemi'
    ];

    const statuses: Array<'active' | 'inactive' | 'pending'> = ['active', 'inactive', 'pending'];
    const data: TableRow[] = [];

    for (let i = 0; i < 15; i++) {
        const hours = Math.floor(Math.random() * 24);
        const minutes = Math.floor(Math.random() * 60);
        const durationMins = Math.floor(Math.random() * 120) + 1;

        data.push({
            userId: `USR-${1000 + i}`,
            activityType: activities[Math.floor(Math.random() * activities.length)],
            time: `${hours.toString().padStart(2, '0')}:${minutes.toString().padStart(2, '0')}`,
            status: statuses[Math.floor(Math.random() * statuses.length)],
            duration: `${durationMins} dk`
        });
    }

    return data;
}

// Populate table
function populateTable() {
    const tableBody = document.getElementById('dataTableBody');
    if (!tableBody) return;

    const data = generateTableData();
    tableBody.innerHTML = '';

    for (const row of data) {
        const tr = document.createElement('tr');
        tr.innerHTML = `
            <td>${row.userId}</td>
            <td>${row.activityType}</td>
            <td>${row.time}</td>
            <td><span class="status-badge status-${row.status}">${getStatusText(row.status)}</span></td>
            <td>${row.duration}</td>
        `;
        tableBody.appendChild(tr);
    }
}

function getStatusText(status: string): string {
    const statusMap: Record<string, string> = {
        'active': 'Aktif',
        'inactive': 'Pasif',
        'pending': 'Beklemede'
    };
    return statusMap[status] || status;
}

// Live stream data
interface LiveItem {
    title: string;
    description: string;
    time: string;
}

function generateLiveItem(): LiveItem {
    const events = [
        { title: 'Yeni Kullanıcı Kaydı', desc: 'Sisteme yeni kullanıcı eklendi' },
        { title: 'Veri İşleme Tamamlandı', desc: 'Toplu veri işleme başarılı' },
        { title: 'Rapor Oluşturuldu', desc: 'Aylık özet raporu hazır' },
        { title: 'Sistem Kontrolü', desc: 'Otomatik sistem kontrolü yapıldı' },
        { title: 'Yedekleme İşlemi', desc: 'Veri yedekleme tamamlandı' },
        { title: 'Performans Optimizasyonu', desc: 'Sistem performansı iyileştirildi' },
        { title: 'Güvenlik Taraması', desc: 'Güvenlik kontrolü başarılı' },
        { title: 'API İsteği', desc: 'Harici API bağlantısı sağlandı' }
    ];

    const event = events[Math.floor(Math.random() * events.length)];
    const now = new Date();
    const time = `${now.getHours().toString().padStart(2, '0')}:${now.getMinutes().toString().padStart(2, '0')}:${now.getSeconds().toString().padStart(2, '0')}`;

    return {
        title: event.title,
        description: event.desc,
        time: time
    };
}

function addLiveItem() {
    const liveStream = document.getElementById('liveStream');
    if (!liveStream) return;

    const item = generateLiveItem();
    const div = document.createElement('div');
    div.className = 'live-item';
    div.innerHTML = `
        <div class="live-item-content">
            <div class="live-item-title">${item.title}</div>
            <div class="live-item-description">${item.description}</div>
        </div>
        <div class="live-item-time">${item.time}</div>
    `;

    liveStream.insertBefore(div, liveStream.firstChild);

    // Keep only the last 10 items
    while (liveStream.children.length > 10) {
        liveStream.removeChild(liveStream.lastChild as Node);
    }
}

// Mobile menu toggle
function initializeMobileMenu() {
    const mobileMenuBtn = document.getElementById('mobileMenuBtn');
    const sidebar = document.getElementById('sidebar');

    if (mobileMenuBtn && sidebar) {
        mobileMenuBtn.addEventListener('click', () => {
            sidebar.classList.toggle('active');
        });

        // Close sidebar when clicking outside
        document.addEventListener('click', (e) => {
            const target = e.target as HTMLElement;
            if (!sidebar.contains(target) && !mobileMenuBtn.contains(target)) {
                sidebar.classList.remove('active');
            }
        });
    }
}

// Filter buttons functionality
function initializeFilterButtons() {
    const filterButtons = document.querySelectorAll('.filter-btn');

    for (const button of filterButtons) {
        button.addEventListener('click', () => {
            for (const btn of filterButtons) {
                btn.classList.remove('active');
            }
            button.classList.add('active');
            // In a real app, this would filter the table data
            populateTable();
        });
    }
}

// Sidebar navigation
function initializeSidebarNavigation() {
    const sidebarLinks = document.querySelectorAll('.sidebar-link');

    for (const link of sidebarLinks) {
        link.addEventListener('click', (e) => {
            e.preventDefault();
            for (const l of sidebarLinks) {
                l.classList.remove('active');
            }
            link.classList.add('active');
        });
    }
}

// Header navigation
function initializeHeaderNavigation() {
    const navLinks = document.querySelectorAll('.nav-link');

    for (const link of navLinks) {
        link.addEventListener('click', (e) => {
            e.preventDefault();
            for (const l of navLinks) {
                l.classList.remove('active');
            }
            link.classList.add('active');
        });
    }
}

// Update summary values periodically
function startSummaryUpdates() {
    setInterval(() => {
        const summaryValues = document.querySelectorAll('.summary-value');
        for (const element of summaryValues) {
            const currentValue = Number.parseFloat(element.textContent?.replace(/[.,]/g, '') || '0');
            const change = Math.floor(Math.random() * 20) - 10; // Random change between -10 and +10
            const newValue = Math.max(0, currentValue + change);
            animateValue(element as HTMLElement, currentValue, newValue, 1000);
        }
    }, 10000); // Update every 10 seconds
}

// Initialize everything when DOM is loaded
document.addEventListener('DOMContentLoaded', () => {
    initializeSummaryCards();
    populateTable();
    initializeMobileMenu();
    initializeFilterButtons();
    initializeSidebarNavigation();
    initializeHeaderNavigation();

    // Add initial live items
    for (let i = 0; i < 5; i++) {
        addLiveItem();
    }

    // Add new live item every 5 seconds
    setInterval(addLiveItem, 5000);

    // Start periodic summary updates
    startSummaryUpdates();
});

export {};
